/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.juneau.microservice.resources;

import static org.apache.juneau.commons.utils.CollectionUtils.*;
import static org.apache.juneau.commons.utils.ThrowableUtils.*;
import static org.apache.juneau.commons.utils.Utils.*;

import java.io.*;
import java.nio.charset.*;
import java.text.*;
import java.util.*;
import java.util.regex.*;

/**
 * Utility class for reading log files.
 *
 * <p>
 * Provides the capability of returning splices of log files based on dates and filtering based on thread and logger
 * names.
 */
public class LogParser implements Iterable<LogParser.Entry>, Iterator<LogParser.Entry>, Closeable {
	/**
	 * Represents a single line from the log file.
	 */
	@SuppressWarnings("javadoc")
	public class Entry {
		public Date date;
		public String severity, logger;
		protected String line, text;
		protected String thread;
		protected List<String> additionalText;
		protected boolean isRecord;

		Entry(String line) throws IOException {
			try {
				this.line = line;
				Matcher m = formatter.getLogEntryPattern().matcher(line);
				if (m.matches()) {
					isRecord = true;
					String s = formatter.getField("date", m);
					if (nn(s))
						date = formatter.getDateFormat().parse(s);
					thread = formatter.getField("thread", m);
					severity = formatter.getField("level", m);
					logger = formatter.getField("logger", m);
					text = formatter.getField("msg", m);
					if (nn(logger) && logger.indexOf('.') > -1)
						logger = logger.substring(logger.lastIndexOf('.') + 1);
				}
			} catch (ParseException e) {
				throw ioex(e);
			}
		}

		public Writer appendHtml(Writer w) throws IOException {
			w.append(toHtml(line)).append("<br>");
			if (nn(additionalText))
				for (var t : additionalText)
					w.append(toHtml(t)).append("<br>");
			return w;
		}

		public String getText() {
			if (additionalText == null)
				return text;
			int i = text.length();
			for (var s : additionalText)
				i += s.length() + 1;
			var sb = new StringBuilder(i);
			sb.append(text);
			for (var s : additionalText)
				sb.append('\n').append(s);
			return sb.toString();
		}

		public String getThread() { return thread; }

		protected Writer append(Writer w) throws IOException {
			w.append(line).append('\n');
			if (nn(additionalText))
				for (var t : additionalText)
					w.append(t).append('\n');
			return w;
		}

		void addText(String t) {
			if (additionalText == null)
				additionalText = new LinkedList<>();
			additionalText.add(t);
		}

		boolean matches() {
			if (! isRecord)
				return false;
			if (nn(start) && date.before(start))
				return false;
			if (nn(end) && date.after(end))
				return false;
			if (nn(threadFilter) && ! threadFilter.equals(thread))
				return false;
			if (nn(loggerFilter) && ! loggerFilter.contains(logger))
				return false;
			if (nn(severityFilter) && ! severityFilter.contains(severity))
				return false;
			return true;
		}
	}

	static String toHtml(String s) {
		if (s.indexOf('<') != -1)
			return s.replaceAll("<", "&lt;");//$NON-NLS-2$
		return s;
	}

	private BufferedReader br;
	LogEntryFormatter formatter;
	Date start, end;
	Set<String> loggerFilter, severityFilter;

	String threadFilter;

	private Entry next;

	/**
	 * Constructor.
	 *
	 * @param formatter The log entry formatter.
	 * @param f The log file.
	 * @param start Don't return rows before this date.  If <jk>null</jk>, start from the beginning of the file.
	 * @param end Don't return rows after this date.  If <jk>null</jk>, go to the end of the file.
	 * @param thread Only return log entries with this thread name.
	 * @param loggers Only return log entries produced by these loggers (simple class names).
	 * @param severity Only return log entries with the specified severity.
	 * @throws IOException Thrown by underlying stream.
	 */
	public LogParser(LogEntryFormatter formatter, File f, Date start, Date end, String thread, String[] loggers, String[] severity) throws IOException {
		br = new BufferedReader(new InputStreamReader(new FileInputStream(f), Charset.defaultCharset()));
		this.formatter = formatter;
		this.start = start;
		this.end = end;
		this.threadFilter = thread;
		if (nn(loggers))
			this.loggerFilter = new LinkedHashSet<>(l(loggers));
		if (nn(severity))
			this.severityFilter = new LinkedHashSet<>(l(severity));

		// Find the first line.
		String line;
		while (next == null && nn(line = br.readLine())) {
			var e = new Entry(line);
			if (e.matches())
				next = e;
		}
	}

	@Override /* Overridden from Closeable */
	public void close() throws IOException {
		br.close();
	}

	@Override /* Overridden from Iterator */
	public boolean hasNext() {
		return nn(next);
	}

	@Override /* Overridden from Iterable */
	public Iterator<Entry> iterator() {
		return this;
	}

	@SuppressWarnings("null")
	@Override /* Overridden from Iterator */
	public Entry next() {
		Entry current = next;
		Entry prev = next;
		try {
			next = null;
			var line = (String)null;
			while (next == null && nn(line = br.readLine())) {
				var e = new Entry(line);
				if (e.isRecord) {
					if (e.matches())
						next = e;
					prev = null;
				} else {
					if (nn(prev))
						prev.addText(e.line);
				}
			}
		} catch (IOException e) {
			throw new UncheckedIOException(e);
		}
		return current;
	}

	@Override /* Overridden from Iterator */
	public void remove() {
		throw new NoSuchMethodError();
	}

	/**
	 * Serializes the contents of the parsed log file to the specified writer and then closes the underlying reader.
	 *
	 * @param w The writer to write the log file to.
	 * @throws IOException Thrown by underlying stream.
	 */
	@SuppressWarnings("resource")
	public void writeTo(Writer w) throws IOException {
		try {
			if (! hasNext())
				w.append("[EMPTY]");
			else
				for (var le : this)
					le.append(w);
		} finally {
			close();
		}
	}
}