/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.sling.models.impl;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.apache.sling.api.SlingJakartaHttpServletRequest;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ValueMap;
import org.apache.sling.api.wrappers.ValueMapDecorator;
import org.apache.sling.models.annotations.Model;
import org.apache.sling.models.annotations.injectorspecific.Self;
import org.apache.sling.models.export.spi.ModelExporter;
import org.apache.sling.models.factory.ExportException;
import org.apache.sling.models.factory.InvalidAdaptableException;
import org.apache.sling.models.factory.MissingElementsException;
import org.apache.sling.models.factory.MissingExporterException;
import org.apache.sling.models.factory.ModelClassException;
import org.apache.sling.models.impl.injectors.SelfInjector;
import org.apache.sling.models.impl.injectors.ValueMapInjector;
import org.apache.sling.models.testmodels.classes.CachedModelWithSelfReference;
import org.apache.sling.models.testmodels.classes.ConstructorWithExceptionModel;
import org.apache.sling.models.testmodels.classes.DefaultStringModel;
import org.apache.sling.models.testmodels.classes.InvalidModelWithMissingAnnotation;
import org.apache.sling.models.testmodels.classes.ResourceModelWithRequiredField;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.junit.jupiter.MockitoExtension;
import org.osgi.framework.BundleContext;
import org.osgi.service.component.ComponentContext;
import org.osgi.util.converter.Converter;
import org.osgi.util.converter.Converters;

import static org.junit.jupiter.api.Assertions.assertDoesNotThrow;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

@ExtendWith(MockitoExtension.class)
public class AdapterFactoryTest {

    @Mock
    private Resource resource;

    @Mock
    private SlingJakartaHttpServletRequest request;

    private ModelAdapterFactory factory;

    public static ModelAdapterFactory createModelAdapterFactory() {
        BundleContext bundleContext = Mockito.mock(BundleContext.class);
        return createModelAdapterFactory(bundleContext);
    }

    public static ModelAdapterFactory createModelAdapterFactory(BundleContext bundleContext) {
        ComponentContext componentCtx = Mockito.mock(ComponentContext.class);
        when(componentCtx.getBundleContext()).thenReturn(bundleContext);

        ModelAdapterFactory factory = new ModelAdapterFactory();
        Converter c = Converters.standardConverter();
        Map<String, String> map = new HashMap<>();
        ModelAdapterFactoryConfiguration config = c.convert(map).to(ModelAdapterFactoryConfiguration.class);
        factory.activate(componentCtx, config);
        factory.injectAnnotationProcessorFactories = Collections.emptyList();
        factory.injectAnnotationProcessorFactories2 = Collections.emptyList();
        factory.injectors = Collections.emptyList();
        factory.implementationPickers = Collections.emptyList();
        return factory;
    }

    @BeforeEach
    void setup() {
        factory = createModelAdapterFactory();
        factory.injectors = Arrays.asList(new ValueMapInjector(), new SelfInjector());
        factory.modelExporters = Arrays.<ModelExporter>asList(
                new FirstStringExporter(), new SecondStringExporter(), new FirstIntegerExporter());
        factory.adapterImplementations.addClassesAsAdapterAndImplementation(
                DefaultStringModel.class,
                ConstructorWithExceptionModel.class,
                NestedModel.class,
                NestedModelWithInvalidAdaptable.class,
                NestedModelWithInvalidAdaptable2.class,
                ResourceModelWithRequiredField.class,
                CachedModelWithSelfReference.class);
    }

    @Test
    void testIsModelClass() {
        assertTrue(factory.isModelClass(DefaultStringModel.class));
        assertFalse(factory.isModelClass(InvalidModelWithMissingAnnotation.class));
    }

    @Test
    void testCanCreateFromAdaptable() {
        assertTrue(factory.canCreateFromAdaptable(resource, DefaultStringModel.class));
        assertFalse(factory.canCreateFromAdaptable(request, DefaultStringModel.class));
    }

    @Test
    void testCanCreateFromAdaptableWithInvalidModel() {
        assertFalse(factory.canCreateFromAdaptable(resource, InvalidModelWithMissingAnnotation.class));
    }

    @Test
    void testCreateFromNonModelClass() {
        assertThrows(
                ModelClassException.class,
                () -> factory.createModel(resource, InvalidModelWithMissingAnnotation.class));
    }

    @Test
    void testCreateFromInvalidAdaptable() {
        assertThrows(InvalidAdaptableException.class, () -> factory.createModel(request, DefaultStringModel.class));
    }

    @Test
    void testCreateWithConstructorException() {
        // Internally all exceptions are wrapped within RuntimeExceptions
        assertThrows(RuntimeException.class, () -> factory.createModel(resource, ConstructorWithExceptionModel.class));
    }

    @Model(adaptables = SlingJakartaHttpServletRequest.class)
    public static class NestedModelWithInvalidAdaptable {
        @Self
        DefaultStringModel nestedModel;
    }

    @Test
    void testCreatedNestedModelWithInvalidAdaptable() {
        // nested model can only be adapted from another adaptable
        assertThrows(
                MissingElementsException.class,
                () -> factory.createModel(request, NestedModelWithInvalidAdaptable.class));
    }

    @Model(adaptables = SlingJakartaHttpServletRequest.class)
    public static class NestedModelWithInvalidAdaptable2 {
        @Self
        InvalidModelWithMissingAnnotation nestedModel;
    }

    @Test
    void testCreatedNestedModelWithInvalidAdaptable2() {
        // nested model is in fact no valid model
        assertThrows(
                MissingElementsException.class,
                () -> factory.createModel(request, NestedModelWithInvalidAdaptable2.class));
    }

    @Model(adaptables = Resource.class)
    public static class NestedModel {
        @Self
        ResourceModelWithRequiredField nestedModel;

        public ResourceModelWithRequiredField getNestedModel() {
            return nestedModel;
        }
    }

    @Test
    void testCreatedNestedModel() {
        Map<String, Object> map = new HashMap<String, Object>();
        map.put("required", "required");
        ValueMap vm = new ValueMapDecorator(map);
        when(resource.adaptTo(ValueMap.class)).thenReturn(vm);

        NestedModel model = factory.createModel(resource, NestedModel.class);
        assertNotNull(model);
        assertEquals("required", model.getNestedModel().getRequired());
    }

    @Test
    void testCreatedNestedModelWithMissingElements() {
        Map<String, Object> map = new HashMap<String, Object>();
        map.put("invalid", "required");
        ValueMap vm = new ValueMapDecorator(map);
        when(resource.adaptTo(ValueMap.class)).thenReturn(vm);

        assertThrows(MissingElementsException.class, () -> factory.createModel(resource, NestedModel.class));
    }

    @Test
    @SuppressWarnings("unchecked")
    void testSelectExporterByName() {
        Result<Object> result = mock(Result.class);
        when(result.wasSuccessful()).thenReturn(true);
        when(result.getValue()).thenReturn(new Object());

        String exported = assertDoesNotThrow(() ->
                factory.handleAndExportResult(result, "second", String.class, Collections.<String, String>emptyMap()));
        assertEquals("Export from second", exported);
    }

    @Test
    @SuppressWarnings("unchecked")
    void testSelectExporterByType() {
        Result<Object> result = mock(Result.class);
        when(result.wasSuccessful()).thenReturn(true);
        when(result.getValue()).thenReturn(new Object());

        Integer exported = assertDoesNotThrow(() ->
                factory.handleAndExportResult(result, "first", Integer.class, Collections.<String, String>emptyMap()));
        assertEquals(Integer.valueOf(42), exported);
    }

    @Test
    @SuppressWarnings("unchecked")
    void testSelectExporterByNameAndWrongType() {
        Result<Object> result = mock(Result.class);
        when(result.wasSuccessful()).thenReturn(true);
        when(result.getValue()).thenReturn(new Object());

        assertThrows(
                MissingExporterException.class,
                () -> factory.handleAndExportResult(
                        result, "second", Integer.class, Collections.<String, String>emptyMap()));
    }

    private static class FirstStringExporter implements ModelExporter {
        @Override
        public boolean isSupported(@NotNull Class<?> aClass) {
            return aClass == String.class;
        }

        @Nullable
        @Override
        @SuppressWarnings("unchecked")
        public <T> T export(@NotNull Object o, @NotNull Class<T> aClass, @NotNull Map<String, String> map)
                throws ExportException {
            if (aClass == String.class) {
                return (T) "Export from first";
            } else {
                throw new ExportException(String.format("%s is not supported.", aClass));
            }
        }

        @NotNull
        @Override
        public String getName() {
            return "first";
        }
    }

    @SuppressWarnings("unchecked")
    private static class SecondStringExporter implements ModelExporter {
        @Override
        public boolean isSupported(@NotNull Class<?> aClass) {
            return aClass == String.class;
        }

        @Nullable
        @Override
        public <T> T export(@NotNull Object o, @NotNull Class<T> aClass, @NotNull Map<String, String> map)
                throws ExportException {
            if (aClass == String.class) {
                return (T) "Export from second";
            } else {
                throw new ExportException(String.format("%s is not supported.", aClass));
            }
        }

        @NotNull
        @Override
        public String getName() {
            return "second";
        }
    }

    @SuppressWarnings("unchecked")
    private static class FirstIntegerExporter implements ModelExporter {
        @Override
        public boolean isSupported(@NotNull Class<?> aClass) {
            return aClass == Integer.class;
        }

        @Nullable
        @Override
        public <T> T export(@NotNull Object o, @NotNull Class<T> aClass, @NotNull Map<String, String> map)
                throws ExportException {
            if (aClass == Integer.class) {
                return (T) Integer.valueOf(42);
            } else {
                throw new ExportException(String.format("%s is not supported.", aClass));
            }
        }

        @NotNull
        @Override
        public String getName() {
            return "first";
        }
    }

    @Test
    void testCreateCachedModelWillNotCrashTheVMWithOOM() {
        /*
         * LOAD_FACTOR is used to ensure the test will try create instances of the model to fill up
         * HEAP_SIZE * LOAD_FACTOR memory. This should be a number > 1.0, to ensure that memory would be
         * exhausted, should this test fail.
         */
        double LOAD_FACTOR = 2.0;
        /*
         * SIZE_OF_LONG is the same as Long.BYTES, in Java 8. When Java 8 source & target are used, this
         * can be removed and replaced with Long.BYTES.
         */
        long SIZE_OF_LONG = 8;
        /*
         * instanceSize is an underestimate of the actual size, based on the number of longs in the array.
         * This works for this test, because we want it to try to OOM the VM, and having more instances is
         * better. The smaller instanceSize is, the more instances we try to create (see the calculation for
         * maxInstances below).
         */
        long instanceSize = SIZE_OF_LONG * CachedModelWithSelfReference.numberOfLongs;
        long maxHeapSize = Runtime.getRuntime().maxMemory();
        long maxInstances = (long) ((maxHeapSize / instanceSize) * LOAD_FACTOR);

        assertDoesNotThrow(() -> {
            for (long i = 0; i < maxInstances; i++) {
                CachedModelWithSelfReference model = factory.createModel(
                        mock(SlingJakartaHttpServletRequest.class), CachedModelWithSelfReference.class);
                for (int j = 0; j < model.longs.length; j++) {
                    model.longs[j] = j;
                }
            }
        });
        // Guard against silent failure: ensure heap is sufficient to run the test loop at least once
        assertTrue(maxInstances > 0);
    }
}
